package furny.ga;

import ga.view.interfaces.IPhenotypeSpace;
import math.geom2d.Point2D;
import math.geom2d.line.LineSegment2D;
import math.geom2d.polygon.Polygon2D;
import math.geom2d.polygon.Rectangle2D;

import com.jme3.math.FastMath;

/**
 * This is a pseudo implementation of a phenotype space. It only calculates
 * bounds virtually.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public class PseudoSpace implements IPhenotypeSpace {
  private final Rectangle2D bounds;

  /**
   * Creates a new pseudo phenotype space.
   * 
   * @param roomWidth
   *          The width in meters.
   * @param roomLength
   *          The length in meters.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public PseudoSpace(final float roomWidth, final float roomLength) {
    final float roomWidthHalf = roomWidth / 2f;
    final float roomLengthHalf = roomLength / 2f;

    bounds = new Rectangle2D(-roomWidthHalf, -roomLengthHalf, roomWidth,
        roomLength);
  }

  @Override
  public boolean contains(final Polygon2D shape) {
    return getOutterBounds().getBoundingBox().containsBounds(shape);
  }

  @Override
  public Rectangle2D getOutterBounds() {
    return bounds;
  }

  @Override
  public float minDistanceToWall(final Polygon2D shape, final float angle) {
    float minDistance = Float.MAX_VALUE;

    // iterate through edges of the room
    for (final LineSegment2D edge : bounds.getEdges()) {

      // iterate through vertices of the bounding shape
      for (final Point2D vert : shape.getVertices()) {

        boolean sameDir = false;

        // this works ok, but assumes that only the back side is allowed to
        // face
        // the wall

        if (Math.abs(angle) < 0.0001f) {
          // assume no rotation

          if (edge.getFirstPoint().getY() < vert.getY()
              && edge.getLastPoint().getY() < vert.getY()) {
            sameDir = true;
          }
        } else if (Math.abs(angle - (FastMath.DEG_TO_RAD * 90f)) < 0.0001f) {
          // assume 90 degrees

          if (edge.getFirstPoint().getX() < vert.getX()
              && edge.getLastPoint().getX() < vert.getX()) {
            sameDir = true;
          }
        } else if (Math.abs(angle - (FastMath.DEG_TO_RAD * 180f)) < 0.0001f) {

          if (edge.getFirstPoint().getY() > vert.getY()
              && edge.getLastPoint().getY() > vert.getY()) {
            sameDir = true;
          }
        } else if (Math.abs(angle - (FastMath.DEG_TO_RAD * 270f)) < 0.0001f) {

          if (edge.getFirstPoint().getX() > vert.getX()
              && edge.getLastPoint().getX() > vert.getX()) {
            sameDir = true;
          }
        }

        if (sameDir) {
          // get the min distance between all vertices and edges
          minDistance = Math.min(minDistance, (float) edge.getDistance(vert));
        }
      }
    }

    return minDistance;
  }
}
